# 機能設計書 46-AIR（Abstract IR）

## 概要

本ドキュメントは、Zigコンパイラの解析済み中間表現AIR（Analyzed Intermediate Representation）について記述する。AIRはSemaによって生成され、コード生成バックエンドによって消費される低レベル中間表現である。

### 本機能の処理概要

AIRは、ZIRから型チェック・意味解析を経て生成される、コード生成直前の中間表現である。ZIRがファイル全体で1インスタンスであるのに対し、AIRは各関数ごとに独立したインスタンスを持つ。

**業務上の目的・背景**：コンパイラパイプラインにおいて、高レベルなZIRと低レベルなマシンコードの間を橋渡しする役割を担う。AIRは型情報が解決済みであり、安全性チェックが挿入された状態となっているため、コード生成バックエンドはAIRを直接機械語に変換できる。複数のバックエンド（LLVM、x86_64、aarch64、WASM等）が共通のAIRを入力として使用することで、コードの重複を避けている。

**機能の利用シーン**：
- Semaによる意味解析結果の保存
- 各種バックエンド（LLVM、ネイティブ）へのコード供給
- インクリメンタルコンパイルでの中間結果キャッシュ
- デバッグ情報生成

**主要な処理内容**：
1. 算術演算命令（add、sub、mul、div等）
2. 制御フロー命令（block、br、loop、repeat）
3. メモリ操作命令（alloc、load、store）
4. 関数呼び出し命令（call）
5. 型変換命令（bitcast、intcast）
6. 安全性チェック命令（safety checks）

**関連システム・外部連携**：
- Sema（入力）：AIRを生成
- codegen（出力）：AIRを消費してネイティブコード生成
- Air/Liveness：AIR命令の生存解析

**権限による制御**：特になし（コンパイラ内部処理）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | AIRは直接的な画面を持たない内部機能 |

## 機能種別

中間表現 / コンパイラ内部処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| instructions | MultiArrayList(Inst).Slice | Yes | AIR命令リスト | 有効な命令列 |
| extra | ArrayList(u32) | Yes | 追加データ | 有効なデータ |

### 入力データソース

- Semaによる意味解析結果

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| machine_code | []u8 | 生成されたマシンコード |
| debug_info | DebugInfo | デバッグ情報 |

### 出力先

各コード生成バックエンド

## 処理フロー

### 処理シーケンス

```
1. Semaによる生成
   └─ ZIR命令をAIR命令に変換
2. Livenessの計算
   └─ 各AIR命令の生存範囲を解析
3. Legalizeの適用
   └─ ターゲット固有の正規化
4. codegenによる消費
   └─ マシンコードへの変換
```

### フローチャート

```mermaid
flowchart TD
    A[ZIR] --> B[Sema]
    B --> C[AIR生成]
    C --> D[Air.Liveness]
    D --> E[Air.Legalize]
    E --> F{バックエンド選択}
    F -->|LLVM| G[LLVM IR]
    F -->|x86_64| H[x86_64コード]
    F -->|aarch64| I[ARM64コード]
    F -->|wasm| J[WebAssembly]
    G --> K[実行可能ファイル]
    H --> K
    I --> K
    J --> L[WASMモジュール]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-46-01 | 関数単位 | AIRは関数ごとに独立 | 全AIR生成時 |
| BR-46-02 | 型解決済み | すべての型は解決済み | AIR命令の型参照 |
| BR-46-03 | SSA形式 | 各命令は一度だけ定義 | 命令参照時 |
| BR-46-04 | 安全性挿入済み | 必要な安全性チェックは挿入済み | 実行時チェック |

### 計算ロジック

AIR命令はSSA（Static Single Assignment）形式で表現され、各命令は一度だけ値を定義する。

## データベース操作仕様

該当なし（メモリ上での処理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidInstruction | 命令エラー | 不正なAIR命令 | Semaのバグ修正 |
| UnsupportedTarget | ターゲットエラー | 未対応ターゲット | 対応ターゲット使用 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- インクリメンタルコンパイルでのAIRキャッシュ活用
- Livenessの効率的な計算

## セキュリティ考慮事項

特になし（コンパイラ内部処理）

## 備考

Air.zigは約104KBのソースコードであり、すべてのAIR命令タグとデータ形式を定義している。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: AIR構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Air.zig | `src/Air.zig` | ファイル先頭コメント（1-6行目） |
| 1-2 | Air.zig | `src/Air.zig` | Air構造体定義（22-26行目） |

**読解のコツ**:
- `instructions: MultiArrayList(Inst).Slice` - 命令リスト
- `extra: ArrayList(u32)` - 追加データ（ペイロード）
- ZIRは1ファイル1インスタンス、AIRは1関数1インスタンス

**主要処理フロー**:
1. **1-6行目**: AIRの役割説明
2. **22行目**: instructionsフィールド
3. **25行目**: extraフィールド

#### Step 2: 命令タグを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Air.zig | `src/Air.zig` | Inst.Tag列挙型（38行目以降） |

**読解のコツ**:
- 算術: add, sub, mul, div等
- 制御フロー: block, br, loop, repeat
- メモリ: alloc, load, store
- 型変換: bitcast, intcast

**主要処理フロー**:
- **43行目**: arg - 関数引数
- **48行目**: add - 加算
- **55行目**: add_safe - 安全性チェック付き加算
- **221行目**: alloc - スタック割り当て
- **284行目**: block - ブロック
- **294行目**: loop - ループ
- **300行目**: br - ブレイク

#### Step 3: 関連モジュールを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Air/Liveness.zig | `src/Air/Liveness.zig` | 生存解析 |
| 3-2 | Air/Legalize.zig | `src/Air/Legalize.zig` | 正規化 |
| 3-3 | Air/print.zig | `src/Air/print.zig` | AIR出力 |

**読解のコツ**:
- Livenessは各命令がどの範囲で使用されるかを解析
- Legalizeはターゲット固有の変換を行う

### プログラム呼び出し階層図

```
Sema.zig (意味解析)
    │
    └─ air_instructions, air_extra に出力
           │
           ▼
       Air構造体
           │
           ├─ Air/Liveness.zig (生存解析)
           ├─ Air/Legalize.zig (正規化)
           │
           └─ codegen.zig
                  │
                  ├─ codegen/llvm.zig (LLVMバックエンド)
                  ├─ codegen/x86_64/ (x86_64バックエンド)
                  ├─ codegen/aarch64/ (ARM64バックエンド)
                  └─ codegen/wasm/ (WASMバックエンド)
```

### データフロー図

```
[入力]                      [処理]                        [出力]

ZIR命令列  ───▶  Sema.analyzeBody()  ───▶  AIR命令列
                      │                        │
                      ▼                        ▼
               型チェック                 instructions[]
               安全性挿入                 extra[]
                      │                        │
                      └────────────────────────┘
                                   │
                                   ▼
                            codegen消費
                                   │
                                   ▼
                            ネイティブコード
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Air.zig | `src/Air.zig` | ソース | AIR命令定義（約104KB） |
| Air/Liveness.zig | `src/Air/Liveness.zig` | ソース | 生存解析 |
| Air/Legalize.zig | `src/Air/Legalize.zig` | ソース | 正規化処理 |
| Air/print.zig | `src/Air/print.zig` | ソース | AIRテキスト出力 |
| Air/types_resolved.zig | `src/Air/types_resolved.zig` | ソース | 型解決チェック |
| Sema.zig | `src/Sema.zig` | ソース | AIR生成元 |
| codegen.zig | `src/codegen.zig` | ソース | AIR消費先 |
